<?php

class DIPI_InstagramProfile extends DIPI_Builder_Module
{
    private static $account_options;

    public $slug = 'dipi_instagram_profile';
    public $vb_support = 'on';

    protected $module_credits = array(
        'module_uri' => 'https://divi-pixel.com/modules/instagram',
        'author' => 'Divi Pixel',
        'author_uri' => 'https://divi-pixel.com',
    );

    public function __construct()
    {
        $account_options = [
            'no_account' => esc_html__('-- Select Account --', 'dipi-divi-pixel'),
        ];
        foreach (\DiviPixel\DIPI_Settings::get_option('facebook_accounts') as $facebook_account_id => $facebook_account) {
            foreach ($facebook_account['instagram_accounts'] as $instagram_account_id => $instagram_account) {
                $account_options["{$facebook_account_id}_{$instagram_account_id}"] = "{$instagram_account['username']} ({$facebook_account['name']})";
            }
        }
        self::$account_options = $account_options;
        parent::__construct();
    }

    public function init()
    {
        $this->name = esc_html__('Pixel Instagram Profile', 'dipi-divi-pixel');
        $this->icon_path = plugin_dir_path(__FILE__) . 'icon.svg';
    }

    public function get_settings_modal_toggles()
    {
        return [
            'general' => [
                'toggles' => [
                    'profile' => esc_html__('Profile Settings', 'dipi-divi-pixel'),
                ],
            ],
            'advanced' => [
                'toggles' => [
                    'profile_text' => [
                        'title' => esc_html__('Text Settings', 'dipi-divi-pixel'),
                        'tabbed_subtoggles' => true,
                        'sub_toggles' => [
                            'username' => array(
                                'name' => esc_html__('UN', 'dipi-divi-pixel'),
                            ),
                            'metrics' => array(
                                'name' => esc_html__('M', 'dipi-divi-pixel'),
                            ),
                            'name' => array(
                                'name' => esc_html__('N', 'dipi-divi-pixel'),
                            ),
                            'biography' => array(
                                'name' => esc_html__('B', 'dipi-divi-pixel'),
                            ),
                            'website' => array(
                                'name' => esc_html__('W', 'dipi-divi-pixel'),
                            ),
                        ],
                    ],
                    'profile_picture' => esc_html__('Profile Picture', 'dipi-divi-pixel'),
                ],
            ],
        ];
    }

    public function get_fields()
    {
        $fields = [];

        if (empty(self::$account_options)) {
            $fields['warning_no_accounts_connected'] = [
                'label' => esc_html__('No Accounts connected', 'dipi-divi-pixel'),
                'message' => sprintf(
                    esc_html__('You haven\'t connected any Facebook accounts yet. Please connect an account via the <a href="%1$s" target="_blank">Divi Pixel settings page</a>.', 'dipi-divi-pixel'),
                    \DiviPixel\DIPI_Settings::admin_url('settings', 'third_party_providers', 'settings_instagram_api_graph')
                ),
                'type' => 'warning',
                'value' => true,
                'display_if' => true,
                'option_category' => 'basic_option',
                'toggle_slug' => 'profile',
            ];

            return $fields;
        }

        $fields['instagram_account'] = [
            'label' => esc_html__('Instagram Account', 'dipi-divi-pixel'),
            'description' => esc_html__('The Instagram account you want to use. Only accounts connected via the graph API work here.', 'dipi-divi-pixel'),
            'type' => 'select',
            'option_category' => 'basic_option',
            'default' => 'no_account',
            'options' => self::$account_options,
            'toggle_slug' => 'profile',
            'computed_affects' => [
                '__profile_html',
            ],
        ];

        $fields['show_profile_picture_url'] = [
            'label' => esc_html__('Show Profile Picture', 'dipi-divi-pixel'),
            'description' => esc_html__('Whether or not to show the profile picture.', 'dipi-divi-pixel'),
            'type' => 'yes_no_button',
            'option_category' => 'layout',
            'default' => 'on',
            'options' => [
                'off' => esc_html__('No', 'dipi-divi-pixel'),
                'on' => esc_html__('Yes', 'dipi-divi-pixel'),
            ],
            'toggle_slug' => 'profile',
            'computed_affects' => [
                '__profile_html',
            ],
        ];

        $fields['show_username'] = [
            'label' => esc_html__('Show Username', 'dipi-divi-pixel'),
            'description' => esc_html__('Whether or not to show the username. The username is the name with which people can tag you.', 'dipi-divi-pixel'),
            'type' => 'yes_no_button',
            'option_category' => 'layout',
            'default' => 'on',
            'options' => [
                'off' => esc_html__('No', 'dipi-divi-pixel'),
                'on' => esc_html__('Yes', 'dipi-divi-pixel'),
            ],
            'toggle_slug' => 'profile',
            'computed_affects' => [
                '__profile_html',
            ],
        ];

        $fields['show_media_count'] = [
            'label' => esc_html__('Show Media Count', 'dipi-divi-pixel'),
            'description' => esc_html__('Whether or not to show the number of images of this account.', 'dipi-divi-pixel'),
            'type' => 'yes_no_button',
            'option_category' => 'layout',
            'default' => 'on',
            'options' => [
                'off' => esc_html__('No', 'dipi-divi-pixel'),
                'on' => esc_html__('Yes', 'dipi-divi-pixel'),
            ],
            'toggle_slug' => 'profile',
            'computed_affects' => [
                '__profile_html',
            ],
        ];

        $fields['show_follower_count'] = [
            'label' => esc_html__('Show Follower Count', 'dipi-divi-pixel'),
            'description' => esc_html__('Whether or not to show the number of accounts who follow this account.', 'dipi-divi-pixel'),
            'type' => 'yes_no_button',
            'option_category' => 'layout',
            'default' => 'on',
            'options' => [
                'off' => esc_html__('No', 'dipi-divi-pixel'),
                'on' => esc_html__('Yes', 'dipi-divi-pixel'),
            ],
            'toggle_slug' => 'profile',
            'computed_affects' => [
                '__profile_html',
            ],
        ];

        $fields['show_following_count'] = [
            'label' => esc_html__('Show Following Count', 'dipi-divi-pixel'),
            'description' => esc_html__('Whether or not to show the number of account this account follows.', 'dipi-divi-pixel'),
            'type' => 'yes_no_button',
            'option_category' => 'layout',
            'default' => 'on',
            'options' => [
                'off' => esc_html__('No', 'dipi-divi-pixel'),
                'on' => esc_html__('Yes', 'dipi-divi-pixel'),
            ],
            'toggle_slug' => 'profile',
            'computed_affects' => [
                '__profile_html',
            ],
        ];

        $fields['show_name'] = [
            'label' => esc_html__('Show Name', 'dipi-divi-pixel'),
            'description' => esc_html__('Whether or not to show the name.', 'dipi-divi-pixel'),
            'type' => 'yes_no_button',
            'option_category' => 'layout',
            'default' => 'on',
            'options' => [
                'off' => esc_html__('No', 'dipi-divi-pixel'),
                'on' => esc_html__('Yes', 'dipi-divi-pixel'),
            ],
            'toggle_slug' => 'profile',
            'computed_affects' => [
                '__profile_html',
            ],
        ];

        $fields['show_biography'] = [
            'label' => esc_html__('Show Biography', 'dipi-divi-pixel'),
            'description' => esc_html__('Whether or not to show the bio.', 'dipi-divi-pixel'),
            'type' => 'yes_no_button',
            'option_category' => 'layout',
            'default' => 'on',
            'options' => [
                'off' => esc_html__('No', 'dipi-divi-pixel'),
                'on' => esc_html__('Yes', 'dipi-divi-pixel'),
            ],
            'toggle_slug' => 'profile',
            'computed_affects' => [
                '__profile_html',
            ],
        ];

        $fields['show_website'] = [
            'label' => esc_html__('Show Website', 'dipi-divi-pixel'),
            'description' => esc_html__('Whether or not to show the website.', 'dipi-divi-pixel'),
            'type' => 'yes_no_button',
            'option_category' => 'layout',
            'default' => 'on',
            'options' => [
                'off' => esc_html__('No', 'dipi-divi-pixel'),
                'on' => esc_html__('Yes', 'dipi-divi-pixel'),
            ],
            'toggle_slug' => 'profile',
            'computed_affects' => [
                '__profile_html',
            ],
        ];

        $fields["__profile_html"] = [
            'type' => 'computed',
            'computed_callback' => ['DIPI_InstagramProfile', 'dipi_get_profile_html'],
            'computed_depends_on' => [
                'instagram_account',
                'show_profile_picture_url',
                'show_username',
                'show_name',
                'show_biography',
                'show_website',
                'show_media_count',
                'show_follower_count',
                'show_following_count',
            ],
            'computed_minimum' => array(
                'instagram_account',
            ),
        ];

        //Design Settings

        $fields['metrics_spacing'] = [
            'label' => esc_html__('Metrics Spacing', 'dipi-divi-pixel'),
            'description' => esc_html__("Spacing between each of the metrics.", 'dipi-divi-pixel'),
            'type' => 'range',
            'option_category' => 'layout',
            'default' => '10px',
            'mobile_options' => true,
            'range_settings' => array(
                'min' => '0',
                'max' => '100',
                'step' => '1',
            ),
            'tab_slug' => 'advanced',
            'toggle_slug' => 'profile_text',
            'sub_toggle' => 'metrics',
        ];

        $fields['profile_picture_size'] = [
            'label' => esc_html__('Size', 'dipi-divi-pixel'),
            'description' => esc_html__("Width and height of the profile picture.", 'dipi-divi-pixel'),
            'type' => 'range',
            'option_category' => 'layout',
            'default' => '150px',
            'mobile_options' => true,
            'range_settings' => array(
                'min' => '0',
                'max' => '500',
                'step' => '1',
            ),
            'tab_slug' => 'advanced',
            'toggle_slug' => 'profile_picture',
        ];

        $fields['profile_picture_spacing'] = [
            'label' => esc_html__('Spacing', 'dipi-divi-pixel'),
            'description' => esc_html__("Spacing between the profile picture and the text. On Desktops and Tablets, the spacing will be added to the right of the image. On Phones it will be added to the bottom.", 'dipi-divi-pixel'),
            'type' => 'range',
            'option_category' => 'layout',
            'default' => '20px',
            'mobile_options' => true,
            'range_settings' => array(
                'min' => '0',
                'max' => '100',
                'step' => '1',
            ),
            'tab_slug' => 'advanced',
            'toggle_slug' => 'profile_picture',
        ];

        //TODO: Border Radius Range Slider
        
        return $fields;
    }

    public function get_advanced_fields_config(){
        $advanced_fields = [];
        $advanced_fields['text'] = false;

        $advanced_fields['fonts']['username'] = [
            'label' => esc_html__('Username', 'dipi-divi-pixel'),
            'toggle_slug' => 'profile_text',
            'sub_toggle' => 'username',
            'css' => [
                'main' => "%%order_class%% .dipi_username",
                'important' => 'all',
            ],
        ];

        $advanced_fields['fonts']['metrics'] = [
            'label' => esc_html__('Metrics', 'dipi-divi-pixel'),
            'toggle_slug' => 'profile_text',
            'sub_toggle' => 'metrics',
            'css' => [
                'main' => "%%order_class%% .dipi_metrics",
                'important' => 'all',
            ],
        ];

        $advanced_fields['fonts']['name'] = [
            'label' => esc_html__('Name', 'dipi-divi-pixel'),
            'toggle_slug' => 'profile_text',
            'sub_toggle' => 'name',
            'css' => [
                'main' => "%%order_class%% .dipi_name",
                'important' => 'all',
            ],
        ];

        $advanced_fields['fonts']['biography'] = [
            'label' => esc_html__('Biography', 'dipi-divi-pixel'),
            'toggle_slug' => 'profile_text',
            'sub_toggle' => 'biography',
            'css' => [
                'main' => "%%order_class%% .dipi_biography",
                'important' => 'all',
            ],
        ];

        $advanced_fields['fonts']['website'] = [
            'label' => esc_html__('Website', 'dipi-divi-pixel'),
            'toggle_slug' => 'profile_text',
            'sub_toggle' => 'website',
            'css' => [
                'main' => "%%order_class%% .dipi_website a",
                'important' => 'all',
            ],
        ];

        return $advanced_fields;
    }

    public static function dipi_get_profile_html($args = [], $conditional_tags = [], $current_page = [])
    {
        // dipi_log("dipi_get_profile_html()", $args, $conditional_tags, $current_page);

        $defaults = [
            'instagram_account' => 'no_account',
            'show_profile_picture_url' => 'on',
            'show_username' => 'on',
            'show_name' => 'on',
            'show_biography' => 'on',
            'show_website' => 'on',
            'show_media_count' => 'on',
            'show_follower_count' => 'on',
            'show_following_count' => 'on',
        ];

        $args = wp_parse_args($args, $defaults);

        $account_components = explode("_", $args['instagram_account']);
        if (count($account_components) < 2 || ($account_components[0] === 'no' && $account_components[1] === 'account')) {
            return sprintf(
                '<div class="dipi_instagram_no_account_warning">%1$s</div>',
                esc_html__('Please select an instagram account.', 'dipi-divi-pixel')
            );
        }

        $facebook_accounts = \DiviPixel\DIPI_Settings::get_option('facebook_accounts');
        $instagram_account = $facebook_accounts[$account_components[0]]['instagram_accounts'][$account_components[1]];
        
        return sprintf(
            '%1$s%2$s',
            self::dipi_get_profile_image($args, $instagram_account),
            self::dipi_get_profile_text($args, $instagram_account)
        );
    }

    private static function dipi_get_profile_image($args, $instagram_account)
    {
        //Create the html for the profile picture, if it should be shown
        if ($args['show_profile_picture_url'] === 'on') {
            //TODO: Add options for alt and title text
            return sprintf(
                '<div class="dipi_profile_picture">
                    <img src="%1$s"/>
                </div>',
                $instagram_account['profile_picture_url']
            );
        } else {
            $profile_picture = '';
        }
    }

    private static function dipi_get_profile_text($args, $instagram_account)
    {
        $content = '<div class="dipi_profile_text">';
        $content .= self::dipi_get_profile_username_and_metrics($args, $instagram_account);
        $content .= self::dipi_get_profile_name($args, $instagram_account);
        $content .= self::dipi_get_profile_bio($args, $instagram_account);
        $content .= self::dipi_get_profile_website($args, $instagram_account);
        $content .= '<div>';
        if ($content === '<div class="dipi_profile_text"><div>') {
            return '';
        } else {
            return $content;
        }
    }

    private static function dipi_get_profile_username_and_metrics($args, $instagram_account)
    {
        $content = '<div class="dipi_username_and_metrics">';

        if ($args['show_username'] === 'on') {
            $content .= sprintf(
                '<div class="dipi_username">%1$s</div>',
                $instagram_account['username']
            );
        }

        $content .= self::dipi_get_count_column($args, $instagram_account);
        $content .= '</div>';
        if ($content === '<div class="username_and_metrics"></div>') {
            return '';
        } else {
            return $content;
        }
    }

    private static function dipi_get_profile_name($args, $instagram_account)
    {
        if ($args['show_name'] === 'on') {
            return sprintf(
                '<div class="dipi_name">%1$s</div>',
                $instagram_account['name']
            );
        } else {
            return '';
        }
    }

    private static function dipi_get_profile_bio($args, $instagram_account)
    {
        if ($args['show_biography'] === 'on') {
            return sprintf(
                '<div class="dipi_biography">%1$s</div>',
                $instagram_account['biography']
            );
        } else {
            return '';
        }
    }

    private static function dipi_get_profile_website($args, $instagram_account){
        if ($args['show_website'] === 'on') {
            return sprintf(
                '<div class="dipi_website"><a href="%1$s">%1$s</a></div>',
                $instagram_account['website']
            );
        } else {
            return '';
        }
    }

    private static function dipi_get_count_column($args, $instagram_account)
    {
        //TODO: Add option to use SVG Images instead of text
        $content = '';

        if ($args['show_media_count'] === 'on') {
            $content .= sprintf(
                '<span class="media_count">%1$s %2$s</span>',
                $instagram_account['media_count'],
                esc_html__('Posts', 'dipi-divi-pixel')
            );
        }

        if ($args['show_follower_count'] === 'on') {
            $content .= sprintf(
                '<span class="followers_count">%1$s %2$s</span>',
                $instagram_account['followers_count'],
                esc_html__('Followers', 'dipi-divi-pixel')
            );
        }

        if ($args['show_following_count'] === 'on') {
            $content .= sprintf(
                '<span class="following_count">%1$s %2$s</span>',
                $instagram_account['follows_count'],
                esc_html__('Following', 'dipi-divi-pixel')
            );
        }

        if ($content === '') {
            return $content;
        } else {
            return sprintf(
                '<div class="dipi_metrics">%1$s</div>',
                $content
            );
        }
    }

    public function render($attrs, $content, $render_slug)
    {
        $this->dipi_apply_css($render_slug);
        return sprintf(
            '<div class="dipi_instagram_profile_wrapper">%1$s</div>',
            self::dipi_get_profile_html($this->props)
        );
    }

    private function dipi_apply_css($render_slug)
    {
        //TODO: Profile Image Size & Border Radius mit mobile options, spacing auf Desktop zwischen Bild und Text
        $this->dipi_apply_profile_picture_css($render_slug);
        
        
        //TODO: dipi_get_profile_username_and_metrics nebeneinander oder untereinander anzeigen mit mobilen Optionen
        
        
        //TODO: Spacing bei Metriken
        $this->dipi_apply_metric_spacing($render_slug);        

        // $css_values = array(
        //     'desktop' => array(
        //         'flex-direction' => esc_html($flex_direction['desktop']),
        //         'justify-content' => esc_html($justify_content['desktop']),
        //         'align-items' => esc_html($align_items['desktop']),
        //         'flex-wrap' => esc_html($flex_wrap_on['desktop'] === 'on' ? 'wrap' : 'nowrap'),
        //     ),
        //     'tablet' => array(
        //         'flex-direction' => esc_html($flex_direction['tablet']),
        //         'justify-content' => esc_html($justify_content['tablet']),
        //         'align-items' => esc_html($align_items['tablet']),
        //         'flex-wrap' => esc_html($flex_wrap_on['tablet'] === 'on' ? 'wrap' : 'nowrap'),
        //     ),
        //     'phone' => array(
        //         'flex-direction' => esc_html($flex_direction['phone']),
        //         'justify-content' => esc_html($justify_content['phone']),
        //         'align-items' => esc_html($align_items['phone']),
        //         'flex-wrap' => esc_html($flex_wrap_on['phone'] === 'on' ? 'wrap' : 'nowrap'),
        //     ),
        // );

        // et_pb_responsive_options()->generate_responsive_css($css_values, '%%order_class%%  .dipi-button-grid-container', '', $render_slug, '', 'flex');

    }

    private function dipi_apply_metric_spacing($render_slug){
        $metrics_spacing = $this->dipi_get_responsive_prop('metrics_spacing');
        et_pb_responsive_options()->generate_responsive_css($metrics_spacing, '%%order_class%% .dipi_metrics span:not(:last-child)', 'margin-right', $render_slug);
    }
    private function dipi_apply_profile_picture_css($render_slug){
        $profile_picture_size = $this->dipi_get_responsive_prop('profile_picture_size');
        et_pb_responsive_options()->generate_responsive_css($profile_picture_size, '%%order_class%% .dipi_profile_picture img', 'width', $render_slug);
        et_pb_responsive_options()->generate_responsive_css($profile_picture_size, '%%order_class%% .dipi_profile_picture img', 'height', $render_slug);
        et_pb_responsive_options()->generate_responsive_css($profile_picture_size, '%%order_class%% .dipi_profile_picture img', 'min-width', $render_slug);
        et_pb_responsive_options()->generate_responsive_css($profile_picture_size, '%%order_class%% .dipi_profile_picture img', 'min-height', $render_slug);
        
        $profile_picture_spacing = $this->dipi_get_responsive_prop('profile_picture_spacing');
        ET_Builder_Element::set_style($render_slug, [
            'selector' => '%%order_class%% .dipi_profile_picture',
            'declaration' => "margin-right: {$profile_picture_size['desktop']}; margin-bottom: 0;",
        ]);
        ET_Builder_Element::set_style($render_slug, [
            'selector' => '%%order_class%% .dipi_profile_picture',
            'declaration' => "margin-right: {$profile_picture_size['tablet']}; margin-bottom: 0;",
            'media_query' => ET_Builder_Element::get_media_query('max_width_980'),
        ]);
        ET_Builder_Element::set_style($render_slug, [
            'selector' => '%%order_class%% .dipi_profile_picture',
            'declaration' => "margin-right: 0; margin-bottom: {$profile_picture_size['phone']};",
            'media_query' => ET_Builder_Element::get_media_query('max_width_767'),
        ]);
    }
}

new DIPI_InstagramProfile();
